# This script provides a PyQt6 GUI to generate instrument sample sets
# from a source WAV file, styled to resemble Apple's design aesthetic.
# Users can select a file via a dialog or by dragging and dropping it.
#
# Required libraries: PyQt6, NumPy, SciPy, and Librosa
# You can install them by running: pip install PyQt6 numpy scipy librosa
#
import sys
import os
import numpy as np
from scipy.io.wavfile import write
import librosa

from PyQt6.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QHBoxLayout, QPushButton,
    QLineEdit, QLabel, QFileDialog, QPlainTextEdit, QMessageBox
)
from PyQt6.QtCore import QObject, QThread, pyqtSignal, Qt
from PyQt6.QtGui import QFont

# ==================================
# APPLE-INSPIRED STYLING (QSS)
# ==================================
def load_stylesheet():
    """Returns a QSS string for styling the application like an Apple product."""
    return """
    QWidget {
        font-family: 'Helvetica Neue', 'Segoe UI', 'Arial', sans-serif;
        font-size: 14px;
        background-color: #f0f0f0;
        color: #333333;
    }
    QLabel {
        background-color: transparent;
    }
    QLineEdit {
        background-color: #ffffff;
        border: 1px solid #cccccc;
        border-radius: 8px;
        padding: 8px 12px;
        font-size: 14px;
    }
    QLineEdit:focus {
        border: 1px solid #007aff;
    }
    QPlainTextEdit {
        background-color: #ffffff;
        border: 1px solid #cccccc;
        border-radius: 8px;
        padding: 8px;
    }
    #LogOutput {
        font-family: 'Menlo', 'Consolas', 'Courier New', monospace;
        font-size: 13px;
        color: #444444;
    }
    QPushButton {
        background-color: #ffffff;
        color: #007aff;
        border: 1px solid #cccccc;
        border-radius: 8px;
        padding: 8px 16px;
        font-weight: 500;
    }
    QPushButton:hover {
        background-color: #f7f7f7;
    }
    QPushButton:pressed {
        background-color: #e0e0e0;
    }
    #GenerateButton {
        background-color: #007aff;
        color: #ffffff;
        border: none;
        font-weight: 600;
    }
    #GenerateButton:hover {
        background-color: #005ecb;
    }
    #GenerateButton:pressed {
        background-color: #004aaa;
    }
    #GenerateButton:disabled {
        background-color: #b0d7ff;
        color: #f0f0f0;
    }
    """

# ==================================
# WORKER THREAD FOR AUDIO PROCESSING
# ==================================
class GenerationWorker(QObject):
    """
    Runs the audio processing on a separate thread to avoid freezing the GUI.
    """
    progress = pyqtSignal(str)
    finished = pyqtSignal()
    error = pyqtSignal(str)

    def __init__(self, source_file, instrument_name, start_octave, num_octaves, sample_rate):
        super().__init__()
        self.source_file = source_file
        self.instrument_name = instrument_name
        self.start_octave = start_octave
        self.num_octaves = num_octaves
        self.sample_rate = sample_rate
        self.is_running = True

    def run(self):
        """The main logic for loading, processing, and saving samples."""
        try:
            # --- 1. Load and Prepare Piano Sample ---
            self.progress.emit(f"Loading sample from: {self.source_file}...")
            piano_base_data, _ = librosa.load(self.source_file, sr=self.sample_rate, mono=True)
            
            self.progress.emit("Detecting base frequency of the sample...")
            f0 = librosa.yin(piano_base_data, fmin=librosa.note_to_hz('C1'), fmax=librosa.note_to_hz('C7'), sr=self.sample_rate)
            valid_f0 = [f for f in f0 if f > 0]
            
            if not valid_f0:
                self.error.emit("Could not detect a stable base frequency for the piano sample.")
                return

            piano_base_freq = np.median(valid_f0)
            self.progress.emit(f"Detected base frequency: {piano_base_freq:.2f} Hz")

            # --- 2. Generate Samples ---
            output_dir = os.path.join('samples', self.instrument_name)
            if not os.path.exists(output_dir):
                os.makedirs(output_dir)
                self.progress.emit(f"Created directory: {output_dir}")

            notes = ['C', 'c', 'D', 'd', 'E', 'F', 'f', 'G', 'g', 'A', 'a', 'B']
            self.progress.emit(f"\nGenerating '{self.instrument_name}' WAV files...")
            
            for octave_num in range(self.start_octave, self.start_octave + self.num_octaves):
                for note_char in notes:
                    filename, data = self._generate_wav_data(
                        note_char, octave_num,
                        piano_sample_data=piano_base_data,
                        piano_base_freq=piano_base_freq
                    )
                    filepath = os.path.join(output_dir, filename)
                    write(filepath, self.sample_rate, data)
                    self.progress.emit(f"   - Saved {filepath}")
            
            self.progress.emit(f"\nSample generation for '{self.instrument_name}' complete!")

        except Exception as e:
            self.error.emit(f"An unexpected error occurred: {e}")
        finally:
            self.finished.emit()

    def _generate_wav_data(self, note_char, octave, piano_sample_data, piano_base_freq, duration=1.5):
        """Generates the audio data for a single note as a NumPy array."""
        base_freq = 261.63
        semitone_map = {'C':0, 'c':1, 'D':2, 'd':3, 'E':4, 'F':5, 'f':6, 'G':7, 'g':8, 'A':9, 'a':10, 'B':11}
        semitones_from_c4 = semitone_map[note_char] + (octave - 4) * 12
        frequency = base_freq * pow(2, semitones_from_c4 / 12)

        data = self._create_piano_waveform_from_sample(
            piano_sample_data, self.sample_rate, piano_base_freq, frequency
        )
        envelope = self._adsr_envelope(duration, self.sample_rate,
                                     attack=0.002, decay=0.3,
                                     sustain_level=0.7, release=0.5)

        min_len = min(len(data), len(envelope))
        data = data[:min_len] * envelope[:min_len]

        amplitude = np.iinfo(np.int16).max * 0.5
        scaled_data = (data * amplitude).astype(np.int16)

        note_name_map = {'C':'C', 'c':'C#', 'D':'D', 'd':'D#', 'E':'E', 'F':'F', 'f':'F#', 'G':'G', 'g':'G#', 'A':'A', 'a':'A#', 'B':'B'}
        filename = f"{note_name_map[note_char]}{octave}.wav"
        return filename, scaled_data

    def _create_piano_waveform_from_sample(self, base_audio, sample_rate, base_freq, target_freq):
        """Generates a piano note by pitch-shifting a base audio sample."""
        if base_freq <= 0:
            return np.zeros_like(base_audio)
        semitones_to_shift = 12 * np.log2(target_freq / base_freq)
        return librosa.effects.pitch_shift(
            y=base_audio, sr=sample_rate, n_steps=semitones_to_shift, bins_per_octave=36
        )

    def _adsr_envelope(self, duration, sample_rate, attack, decay, sustain_level, release):
        """Creates an ADSR (Attack, Decay, Sustain, Release) envelope."""
        total_samples = int(duration * sample_rate)
        attack_samples = int(attack * sample_rate)
        decay_samples = int(decay * sample_rate)
        release_samples = int(release * sample_rate)
        sustain_samples = total_samples - attack_samples - decay_samples - release_samples
        if sustain_samples < 0: sustain_samples = 0
        
        attack_env = np.linspace(0, 1, attack_samples)
        decay_env = np.linspace(1, sustain_level, decay_samples)
        sustain_env = np.full(sustain_samples, sustain_level)
        release_env = np.linspace(sustain_level, 0, release_samples)
        
        # Ensure the envelope does not exceed total_samples
        full_env = np.concatenate((attack_env, decay_env, sustain_env, release_env))
        return full_env[:total_samples]

# =================
# MAIN GUI WINDOW
# =================
class SampleGeneratorApp(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("WAV Sample Set Generator")
        self.setMinimumSize(600, 450)
        self.thread = None
        self.worker = None
        self.setAcceptDrops(True) # Enable drag-and-drop
        self._init_ui()

    def _init_ui(self):
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(20, 20, 20, 20)
        main_layout.setSpacing(15)

        # --- Source File Selection ---
        file_layout = QHBoxLayout()
        file_layout.addWidget(QLabel("Source WAV File:"))
        self.file_path_edit = QLineEdit()
        self.file_path_edit.setReadOnly(True)
        self.file_path_edit.setPlaceholderText("Drag & drop a .wav file here, or click 'Browse...'")
        file_layout.addWidget(self.file_path_edit)
        browse_button = QPushButton("Browse...")
        browse_button.clicked.connect(self.select_source_file)
        file_layout.addWidget(browse_button)
        main_layout.addLayout(file_layout)
        
        # --- Output Instrument Name ---
        name_layout = QHBoxLayout()
        name_layout.addWidget(QLabel("New Instrument Name:"))
        self.instrument_name_edit = QLineEdit()
        self.instrument_name_edit.setPlaceholderText("e.g., piano3, custom_synth")
        name_layout.addWidget(self.instrument_name_edit)
        main_layout.addLayout(name_layout)

        # --- Generate Button ---
        self.generate_button = QPushButton("Generate Samples")
        self.generate_button.setObjectName("GenerateButton") # For specific styling
        self.generate_button.clicked.connect(self.start_generation)
        self.generate_button.setFixedHeight(40)
        main_layout.addWidget(self.generate_button)

        # --- Log Output ---
        main_layout.addWidget(QLabel("Log:"))
        self.log_output = QPlainTextEdit()
        self.log_output.setObjectName("LogOutput") # For specific styling
        self.log_output.setReadOnly(True)
        self.log_output.setPlaceholderText("Processing steps will appear here...")
        main_layout.addWidget(self.log_output)

    # --- DRAG AND DROP EVENT HANDLERS ---
    def dragEnterEvent(self, event):
        """Handles a drag event entering the widget area."""
        mime_data = event.mimeData()
        # Check if the dragged data contains URLs and if one is a local .wav file
        if mime_data.hasUrls():
            for url in mime_data.urls():
                if url.isLocalFile() and url.toLocalFile().lower().endswith('.wav'):
                    event.acceptProposedAction()
                    return
        event.ignore()

    def dragMoveEvent(self, event):
        """This can re-use the enter event logic."""
        self.dragEnterEvent(event)

    def dropEvent(self, event):
        """Handles the user dropping a file."""
        mime_data = event.mimeData()
        if mime_data.hasUrls():
            # Find the first valid .wav file and use it
            for url in mime_data.urls():
                file_path = url.toLocalFile()
                if url.isLocalFile() and file_path.lower().endswith('.wav'):
                    self.file_path_edit.setText(file_path)
                    event.acceptProposedAction()
                    return # Stop after accepting the first valid file
        event.ignore()

    def select_source_file(self):
        """Opens a file dialog to select a .wav file."""
        file_name, _ = QFileDialog.getOpenFileName(
            self, "Select Source Audio File", "", "WAV Files (*.wav)"
        )
        if file_name:
            self.file_path_edit.setText(file_name)

    def start_generation(self):
        """Validates inputs and starts the background worker thread."""
        source_file = self.file_path_edit.text()
        instrument_name = self.instrument_name_edit.text().strip()

        if not source_file:
            QMessageBox.warning(self, "Input Missing", "Please select a source WAV file.")
            return
        if not instrument_name:
            QMessageBox.warning(self, "Input Missing", "Please enter a name for the new instrument.")
            return
        
        self.generate_button.setEnabled(False)
        self.log_output.clear()
        self.log_output.appendPlainText("Starting generation process...")

        # Create and start the worker thread
        self.thread = QThread()
        self.worker = GenerationWorker(
            source_file=source_file,
            instrument_name=instrument_name,
            start_octave=3,
            num_octaves=2,
            sample_rate=44100
        )
        self.worker.moveToThread(self.thread)

        # Connect signals and slots
        self.thread.started.connect(self.worker.run)
        self.worker.finished.connect(self.thread.quit)
        self.worker.finished.connect(self.worker.deleteLater)
        self.thread.finished.connect(self.thread.deleteLater)
        self.worker.progress.connect(self.update_log)
        self.worker.error.connect(self.on_error)
        self.thread.finished.connect(self.on_finished)

        self.thread.start()

    def update_log(self, message):
        """Appends a message from the worker to the log."""
        self.log_output.appendPlainText(message)

    def on_error(self, error_message):
        """Shows an error message and cleans up."""
        QMessageBox.critical(self, "Error", error_message)
        self.log_output.appendPlainText(f"\nERROR: {error_message}")
        self.on_finished()

    def on_finished(self):
        """Re-enables the button and cleans up the thread."""
        self.generate_button.setEnabled(True)
        self.thread = None
        self.worker = None

if __name__ == "__main__":
    app = QApplication(sys.argv)
    # Set the stylesheet for the entire application
    app.setStyleSheet(load_stylesheet())
    
    window = SampleGeneratorApp()
    window.show()
    sys.exit(app.exec())